/*  Copyright (C) 2019 CZ.NIC, z.s.p.o. <knot-dns@labs.nic.cz>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <assert.h>
#include <string.h>

#include "libdnssec/error.h"
#include "libdnssec/keyid.h"
#include "libdnssec/shared/shared.h"

#include "contrib/ctype.h"
#include "contrib/tolower.h"

/* -- public API ----------------------------------------------------------- */

_public_
bool dnssec_keyid_is_valid(const char *id)
{
	if (!id) {
		return false;
	}

	size_t id_len = strlen(id);
	bool with_colons = (id_len > 3 && id[2] == ':'); // p11tool format XX:XX:XX:<..>:XX
	if ((id_len - (with_colons ? id_len / 3 : 0)) % 2 != 0) {
		return false;
	}

	for (int i = 0; id[i] != '\0'; i++) {
		if (with_colons && i > 0 && i < id_len - 1 && i % 3 == 2 && id[i] == ':') {
			continue;
		}
		if (!is_xdigit(id[i])) {
			return false;
		}
	}

	return true;
}

_public_
void dnssec_keyid_normalize(char *id)
{
	if (!id) {
		return;
	}

	size_t j = 0;
	for (size_t i = 0; id[i] != '\0'; i++) {
		if (id[i] == ':') {
			assert(i % 3 == 2);
			continue;
		}
		assert(id[i] != '\0' && is_xdigit(id[i]));
		id[j++] = knot_tolower(id[i]);
	}
	id[j] = '\0';
}

_public_
char *dnssec_keyid_copy(const char *id)
{
	if (!id) {
		return NULL;
	}

	char *copy = strdup(id);
	if (!copy) {
		return NULL;
	}

	dnssec_keyid_normalize(copy);

	return copy;
}

_public_
bool dnssec_keyid_equal(const char *one, const char *two)
{
	if (!one || !two) {
		return NULL;
	}

	return (strcasecmp(one, two) == 0);
}
